"use client";
import { FormFieldType } from "@/types/FormFieldType";
import FormField from "./form-field/FormField";
import { useState } from "react";
import { egyptGovernorates } from "@/data/egyptGovernorates";
import { OrderInputs } from "@/types/OrderInputs";

const formFields: FormFieldType[] = [
  { placeholder: "الاسم", type: "text", name: "name" },
  // { placeholder: "Email Address", type: "text", name: "email" },
  { placeholder: "رقم التليفون", type: "text", name: "phone" },
];

type FieldNames = "name" | "phone";

const intialOrderValues: OrderInputs = {
  name: "",
  phone: "",
  address: "",
  governorate: "",
};
export default function OrderForm({
  quantity,
  coarseDegree,
  price,
  close,
}: {
  quantity: number;
  coarseDegree: string;
  price: number;
  close: () => void;
}) {
  const [formInputs, setFormInputs] = useState<OrderInputs>(intialOrderValues);
  const [error, setError] = useState<OrderInputs>({} as OrderInputs);

  const [status, setStatus] = useState<{
    type: "success" | "failed" | "pending" | "initial";
    message: string;
  }>({ type: "initial", message: "" });

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    let isValid = true;
    if (!formInputs.phone) {
      setError((prev) => ({
        ...prev,
        phone: "من فضلك قم بإدخال رقم الهاتف",
      }));
      isValid = false;
    } else if (!/^(?:\+02)?01[0-9\u0660-\u0669]{9}$/.test(formInputs.phone)) {
      setError((prev) => ({
        ...prev,
        phone: "قم بإدخال رقم هاتف صحيح",
      }));
      isValid = false;
    } else {
      setError((prev) => ({
        ...prev,
        phone: "",
      }));
    }
    if (!formInputs.name) {
      setError((prev) => ({
        ...prev,
        name: "من فضلك قم بإدخال اسمك",
      }));
      isValid = false;
    } else if (
      !formInputs.name.match(/^([a-zA-Z\s]{1,50}|[\u0600-\u06FF\s]{1,50})$/)
    ) {
      setError((prev) => ({
        ...prev,
        name: "الاسم يجب أن يحتوي على حروف عربية أو إنجليزية فقط، وبحد أقصى ٥٠ حرفًا",
      }));
      isValid = false;
    } else {
      setError((prev) => ({
        ...prev,
        name: "",
      }));
    }

    if (!formInputs.governorate) {
      setError((prev) => ({ ...prev, governorate: "من فضلك اختر المحافظة" }));
      isValid = false;
    } else {
      setError((prev) => ({ ...prev, governorate: "" }));
    }
    if (!isValid) return;
    setError({ ...intialOrderValues });
    const data = {
      price,
      quantity,
      coarseDegree,
      ...formInputs,
    };
    setStatus({ type: "pending", message: "" });
    const response = await fetch("/api/send-email", {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify(data),
    });

    if (response.ok) {
      setStatus({ type: "success", message: "تم إرسال الطلب بنجاح" });
    } else {
      setStatus({ type: "failed", message: "برجاء المحاولة مرة أخري" });
    }
    setFormInputs({ ...intialOrderValues });
  };

  return (
    <div className="fixed inset-0 z-20 grid h-screen w-full place-content-center">
      <div
        className="overlay bg-black bg-opacity-40"
        onClick={() => {
          if (status.type !== "pending") close();
        }}
      ></div>
      <div className="container relative z-40  min-h-[200px] w-screen  max-w-[450px]">
        <div className="  size-full place-content-center rounded-md bg-white p-5">
          {status.message ? (
            <p
              className={`w-full rounded-md p-2 text-center text-white ${status.type === "success" && "bg-green-500"} ${status.type === "failed" && "bg-red-500"}`}
            >
              {status.message}
            </p>
          ) : (
            <form className="flex flex-col gap-5" onSubmit={handleSubmit}>
              {formFields.map((field, i) => (
                <FormField
                  key={i}
                  type={field.type}
                  as={field.as}
                  placeholder={field.placeholder || ""}
                  className={field.className}
                  value={formInputs[field.name as FieldNames]}
                  handleChange={(
                    e: React.ChangeEvent<
                      HTMLInputElement | HTMLTextAreaElement
                    >,
                  ) =>
                    setFormInputs((prev) => ({
                      ...prev,
                      [field.name]: e.target.value,
                    }))
                  }
                  error={error}
                  name={field.name as FieldNames}
                />
              ))}
              <div>
                <select
                  name="governorate"
                  id="governorate"
                  onChange={(e) =>
                    setFormInputs((prev) => ({
                      ...prev,
                      governorate: e.target.value,
                    }))
                  }
                  value={formInputs.governorate}
                  className={`w-full rounded-md border border-s-4 border-slate-300 py-3 ps-5 shadow-sm  placeholder:text-slate-400 focus:border-secondary-500 focus:outline-none focus:ring-1 focus:ring-secondary-500 `}
                >
                  <option value="" hidden>
                    اختر المحافظة
                  </option>
                  {egyptGovernorates.map((gov) => (
                    <option value={gov.governorate_name_ar} key={gov.id}>
                      {gov.governorate_name_ar}
                    </option>
                  ))}
                </select>
                {error.governorate && (
                  <p className="mt-2 text-sm text-red-700">
                    {error.governorate}
                  </p>
                )}
              </div>
              <FormField
                key="العنوان"
                placeholder="العنوان"
                value={formInputs.address}
                handleChange={(
                  e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>,
                ) =>
                  setFormInputs((prev) => ({
                    ...prev,
                    address: e.target.value,
                  }))
                }
                error={error}
                name="address"
              />

              <button
                className={`rounded-md bg-secondary-500 py-3 text-center text-lg text-white transition-all duration-300 ease-in-out hover:bg-secondary-400 disabled:bg-slate-300 disabled:text-slate-700`}
                type="submit"
                disabled={status.type === "pending"}
              >
                {status.type === "pending" ? (
                  <div className="flex justify-center gap-2">
                    <span>جاري تأكيد الطلب</span>
                    <span className="loader block"></span>
                  </div>
                ) : (
                  <span>تأكيد الطلب</span>
                )}
              </button>
            </form>
          )}
        </div>
      </div>
    </div>
  );
}
